{********************************************************}
{                                                        }
{       Y-Components Library                             }
{       Lookup control                                   }
{                                                        }
{       Written by:                                      }
{         Igor Tugay                                     }
{         NS System                                      }
{         e-mail:     igor@novell.kz                     }
{         instruction: see end of file                   }
{********************************************************}
unit YDBLook;

// !!!!Do not forget to specify the path to DsgnIntf unit.
// Usually it located in C:\DELPHI5\SOURCE\TOOLSAPI\

interface

uses Windows, Classes, Controls, Graphics, SysUtils, Messages,
     StdCtrls, ExtCtrls, Forms, DsgnIntf,
     DB, DBTables, DBCtrls, YDBGrid, DBGrids;

type
     TFindMode = (fmInclude,fmMatch,fmStart);

//****************************************************
//***                                              ***
//***       TYDBLookUp                             ***
//***                                              ***
//****************************************************
     TYDBLookUp = class(TCustomControl)
     protected
       fDataLink     : TFieldDataLink;
       fDatabaseName : string;
       fSQLShow      : TStrings;
       fSQLFind      : TStrings;
       fFindMode     : TFindMode;
       fFindUpCase   : boolean;
       fBtnWidth     : integer;
       fAdjustCols   : boolean;
       fValue        : string;
       fLastValue    : string;
       fText         : string;
       fOnChange    : TNotifyEvent;
       fOnClickAdd  : TNotifyEvent;
       fOnClickSel  : TNotifyEvent;
       fAfterSelect : TNotifyEvent;
       fLockChange  : integer;
     private
       procedure SetCtl3D(V : boolean);
       function  GetCtl3D : boolean;
       procedure SetBorderStyle(aStyle : TBorderStyle);
       function  GetBorderStyle : TBorderStyle;
       procedure DrawBtn(aIndex : integer; aPressed : boolean; aChar : char);
       function  BtnCount : integer;
       function  BtnRect(aIndex : integer) : TRect;

       procedure CMGetDataLink(var Message: TMessage); message CM_GETDATALINK;

       function  GetDataSource : TDataSource;
       procedure SetDataSource(aDS : TDataSource);
       function  GetDataField : string;
       procedure SetDataField(aDataField : string);
       procedure SetDatabaseName(const V: string);
       procedure SetValue(V : string);
       procedure DataChange(Sender : TObject);
       function  ShowDataOf(V : string) : string;
       procedure SetSQLShow(V : TStrings);
       procedure SetSQLFind(V : TStrings);
     public
       Edit : TEdit;
       constructor Create(aOwner : TComponent); override;
       destructor  Destroy; override;
       procedure   Loaded; override;
       procedure   Resize; override;
       procedure   DoExit; override;
       procedure   Paint; override;
       procedure   SetEnabled(V : boolean); override;
       procedure   SetFocus; override;
       procedure   Notification(AComponent: TComponent; Operation: TOperation); override;
       procedure   MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
       procedure   xKeyPress(Sender: TObject; var Key: Char);
       procedure   xKeyDown(Sender : TObject; var Key: Word; Shift: TShiftState);
       procedure   SelectRecord;
       procedure   Reload;

       procedure   DoClickSel; virtual;
       procedure   DoClickAdd; virtual;

       property    Value : string read fValue write SetValue;
     published
       property Align;
       property Font;
       property TabStop;
       property TabOrder;
       property Enabled;
       property Visible;
       property Ctl3D : boolean read GetCtl3D write SetCtl3D;
       property BorderStyle : TBorderStyle read GetBorderStyle write SetBorderStyle;
       property DataSource : TDataSource read GetDataSource write SetDataSource;
       property DataField : string read GetDataField write SetDataField;
       property DatabaseName: string read FDatabaseName write SetDatabaseName;
       property AdjustCols  : boolean read fAdjustCols write fAdjustCols;
       property SQLShow     : TStrings read fSQLShow write SetSQLShow;
       property SQLFind     : TStrings read fSQLFind write SetSQLFind;
       property SQLFindMode  : TFindMode read fFindMode write fFindMode;
       property SQLFindUpCase: boolean read fFindUpCase write fFindUpCase;
       property AfterSelect  : TNotifyEvent read fAfterSelect write fAfterSelect;
       property OnChange     : TNotifyEvent read fOnChange write fOnChange;
       property OnClickAdd   : TNotifyEvent read fOnClickAdd write fOnClickAdd;
       property OnClickSel   : TNotifyEvent read fOnClickSel write fOnClickSel;
     end;

  TDatabaseNameProperty = class(TStringProperty)
  public
    function  GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
  end;

//****************************************************
//***                                              ***
//***       TYSelForm                              ***
//***                                              ***
//****************************************************
  TYSelForm = class(TForm)
  public
     DS  : TDataSource;
     DG  : TYDBGrid;
     VY  : TYDBLookUp;
     ColWidths : array[0..255] of integer;
     constructor Create(V : TComponent); override;
     destructor  Destroy; override;
     procedure   FormDeactivate(Sender: TObject);
     procedure   DGKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
     procedure   DGDblClick(Sender: TObject);
     procedure   DSDataChange(Sender: TObject; Field: TField);
     procedure   AdjustColWidths;
     function    Run(V : TYDBLookUp; X : TDataSet) : integer;
  end;

procedure Register;

implementation

{$R *.DCR}

procedure Register;
begin
  RegisterComponents('YDB',[TYDBLookUp]);
  RegisterPropertyEditor(TypeInfo(string), TYDBLookUp, 'DatabaseName', TDatabaseNameProperty);
end;

function TDatabaseNameProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList, paSortList, paMultiSelect];
end;

procedure TDatabaseNameProperty.GetValues(Proc: TGetStrProc);
var
  I: Integer;
  Values: TStringList;
begin
  Values := TStringList.Create;
  try
    Session.GetDatabaseNames(Values);
    for I := 0 to Values.Count - 1 do Proc(Values[I]);
  finally
    Values.Free;
  end;
end;

function UnpackFields(Q : TDataSet; SkipFirst : boolean) : string;
var I : integer;
    F : TField;
    S : string;
begin
   result:='';
   for I:=integer(SkipFirst) to Q.FieldCount-1 do
   begin
      F:=Q.Fields[I];
      case F.DataType of
      ftFloat,
      ftCurrency : S:=FormatFloat('### ##0.#0',F.AsFloat);
      ftDate,
      ftDateTime : S:=FormatDateTime('dd-mmm-yyyy',F.AsDateTime);
      ftTime     : S:=FormatDateTime('hh:mm:ss',F.AsDateTime);
      else         S:=F.AsString;
      end;
      result:=result+S;
      if I<> Q.FieldCount-1 then result:=result+', ';
   end;
end;

//****************************************************
//***                                              ***
//***       TYSelForm                              ***
//***                                              ***
//****************************************************
constructor TYSelForm.Create(V : TComponent);
begin
   CreateNew(V);
   Height:=150;
   BorderStyle:=bsNone;
   Ctl3D:=false;
   DS:=TDataSource.Create(Self);
   DS.OnDataChange:=DSDataChange;

   DG:=TYDBGrid.Create(Self);
   DG.y_AutoLocate:=true;
   DG.options:=DG.Options-[dgEditing,dgTitles,dgRowLines,dgIndicator];
   DG.Align:=alClient;
   DG.OnKeyDown:=DGKeyDown;
   DG.OnDblClick:=DGDblClick;
   DG.DataSource:=DS;

   InsertControl(DG);

   onDeactivate:=FormDeactivate;
end;

//****************************************************
destructor  TYSelForm.Destroy;
begin
   DS.Free;
   inherited;
end;

//****************************************************
procedure   TYSelForm.FormDeactivate(Sender: TObject);
begin
   ModalResult:=mrCancel;
   inherited;
end;

//****************************************************
procedure   TYSelForm.DGKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
   if Key = vk_Return then ModalResult:=mrOK else
   if Key = vk_Escape then ModalResult:=mrCancel;
end;

//****************************************************
procedure   TYSelForm.DGDblClick(Sender: TObject);
begin
   ModalResult:=mrOK;
end;

procedure   TYSelForm.DSDataChange(Sender: TObject; Field: TField);
begin
  if (VY <> nil)
  and (DS.DataSet <> nil)
  then VY.Edit.Text:=UnpackFields(DS.DataSet,true);
end;

procedure   TYSelForm.AdjustColWidths;
var I,N : integer;
    X   : TDataSet;
    B   : TBookMark;
begin
   X:=DS.DataSet;
   if X.FieldCount < 3 then exit;

   X.DisableControls;
   Canvas.Font.Assign(VY.Edit.Font);
   FillChar(ColWidths,SizeOf(ColWidths),0);
   B:=X.GetBookMark;
   X.First;
   while not X.EOF do
   begin
      for I:=1 to X.Fields.Count-1 do
      begin
         N:=Canvas.TextWidth(X.Fields[I].DisplayText)+4;
         if N > ColWidths[I] then ColWidths[I]:=N;
      end;
      X.Next;
   end;
   X.GoToBookmark(B);
   X.FreeBookmark(B);
   X.EnableControls;
end;

//****************************************************
function TYSelForm.Run(V : TYDBLookUp; X : TDataSet) : integer;
var R : TPoint;
    I : integer;
begin
   VY:=V;
   DS.DataSet := X;
   DG.Font.Assign(V.Font);

   Width:=V.Width-1;
   R.X:=0; R.Y:=0; R:=V.ClientToScreen(R);
   Left:=R.X;
   Top :=R.Y+V.Height+2;

   if DG.Columns.Count = 1 then DG.Columns[0].Width:=Width-18 else

   if V.AdjustCols then
   begin
     AdjustColWidths;
     for I:=1 to X.Fields.Count-1 do
         DG.Columns[I-1].Width:=ColWidths[I];
   end;


   if Top+Height > Screen.Height
      then Top:=Top-Height-V.Height-2;

   ModalResult:=mrNone;
   Show;
   while (ModalResult=mrNone) and Visible do
   Application.HandleMessage;
   result:=ModalResult;
   Close;
   DS.DataSet:=nil;
   VY:=nil;
end;



const DefBtnWidth = 17;
var   YSelForm    : TYSelForm;

//****************************************************
//***                                              ***
//***       TYDBLookUp                             ***
//***                                              ***
//****************************************************
constructor TYDBLookUp.Create(aOwner : TComponent);
begin
  inherited;
  fBtnWidth := DefBtnWidth;
  fDataLink := TFieldDataLink.Create;
  fDataLink.OnDataChange:=DataChange;

  fSQLShow:=TStringList.Create;
  fSQLFind:=TStringList.Create;

  Height:=21;
  Width:=125;
  Edit:=TEdit.Create(Self);
  Edit.AutoSize:=false;
  Edit.OnKeyPress:=xKeyPress;
  Edit.OnKeyDown:=xKeyDown;
  InsertControl(Edit);
end;

destructor  TYDBLookUp.Destroy;
begin
   fSQLShow.Free;
   fSQLFind.Free;
   fDataLink.Free;

   Edit.Free;
   inherited;
end;

procedure   TYDBLookUp.Loaded;
begin
  inherited;
  Resize;
end;

procedure TYDBLookUp.CMGetDataLink(var Message: TMessage);
begin
  //Message.Result := Integer(FDataLink);
end;

procedure TYDBLookUp.DoExit;
begin
   if Edit.Text <> fText then SelectRecord else inherited;
   Edit.SelLength:=0;
end;



procedure TYDBLookUp.SetCtl3D(V : boolean);
begin
   inherited Ctl3D:=V;
   if V
      then fBtnWidth := DefBtnWidth
      else fBtnWidth := DefBtnWidth-2;
   Resize;
   Invalidate;
end;

function  TYDBLookUp.GetCtl3D : boolean;
begin
  result := inherited Ctl3D;
end;

procedure TYDBLookUp.SetEnabled(V : boolean);
begin
  inherited;
  Resize;
  Invalidate;
end;

procedure TYDBLookUp.SetFocus;
begin
   inherited;
   Edit.SetFocus;
end;

procedure TYDBLookUp.SetBorderStyle(aStyle : TBorderStyle);
begin
  Edit.BorderStyle:=aStyle;
  Resize;
  Invalidate;
end;

function  TYDBLookUp.GetBorderStyle : TBorderStyle;
begin
  result:=Edit.BorderStyle
end;

procedure TYDBLookUp.Resize;
var R : TRect;
begin
  R:=GetClientRect;
  Edit.Left:=0;
  Edit.Top:=0;
  Edit.Width:=R.Right-BtnCount*fBtnWidth-Integer(BorderStyle=bsNone);
  Edit.Height:=R.Bottom;
end;

procedure TYDBLookUp.Paint;
begin
  DrawBtn(0,false,'>');
  DrawBtn(1,false,'+');
  inherited;
end;

function  TYDBLookUp.BtnCount : integer;
begin
  if Enabled then
  begin
    result:=1;
    if Assigned(fOnClickAdd)
       then inc(result);
  end else result:=0;
end;

function  TYDBLookUp.BtnRect(aIndex : integer) : TRect;
begin
   result:=GetClientRect;
   result.Left:=result.Right - BtnCount*fBtnWidth + aIndex*fBtnWidth-1;
   result.Right:=result.Left+fBtnWidth+Integer(Ctl3D=false)+Integer(BorderStyle=bsNone);
end;

procedure TYDBLookUp.DrawBtn(aIndex : integer; aPressed : boolean; aChar : char);
var R   : TRect;
    I,C,H : integer;
begin
   R:=BtnRect(aIndex);
   if Ctl3d
      then Frame3D(Canvas,R,clGray,clWhite,1);
   if Edit.BorderStyle=bsSingle
      then Frame3D(Canvas,R,clBlack,clBlack,1)
      else InflateRect(R,-1,0);
   if aPressed
      then Frame3D(Canvas,R,clGray,clWhite,1)
      else Frame3D(Canvas,R,clWhite,clGray,1);
   C:=(R.Right+R.Left) div 2;
   H:=(R.Bottom+R.Top) div 2;
   Canvas.Pen.Color:=clBlack;

   if aChar = '>' then
   begin
     for I:=0 to 3 do
     begin
       Canvas.MoveTo(C-I,H-I+2);
       Canvas.LineTo(C+I+1,H-I+2);
     end;
   end else
   if aChar = '+' then
   begin
     for I:=0 to 1 do
     begin
       Canvas.MoveTo(C-3,H+I-1);
       Canvas.LineTo(C+3,H+I-1);

       Canvas.MoveTo(C+I-1,H+2);
       Canvas.LineTo(C+I-1,H-4);
     end;
   end;
end;

function  TYDBLookUp.GetDataSource : TDataSource;
begin
  result:=fDataLink.DataSource;
end;

procedure TYDBLookUp.SetDataSource(aDS : TDataSource);
begin
  fDataLink.DataSource:=aDS;
end;

procedure TYDBLookUp.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited;
  if (Operation = opRemove)
  and (AComponent = DataSource)
    then DataSource:=nil;
end;

procedure TYDBLookUp.SelectRecord;
label L0;
var S : string;
    I : integer;
    Selected : boolean;
    Mask : string;
    X    : TQuery;
begin
   if fText = Edit.Text
      then Mask := ''
      else Mask := Edit.Text;

   Selected:=false;
   X:=TQuery.Create(nil);
L0:
   S:=fSQLfind.Text;
   if fFindUpCase
      then Mask:=AnsiUpperCase(Trim(Mask))
      else Mask:=Trim(Mask);

   while Pos('#',S) <> 0 do
   begin
      I:=Pos('#',S);
      Delete(S,I,1);
      case fFindMode of
      fmInclude : Insert('%'+Mask+'%',S,I);
      fmMatch   : Insert(Mask,S,I);
      fmStart   : Insert(Mask+'%',S,I);
      end;
   end;

   if fDatabaseName <> ''
   then X.DatabaseName:=fDatabaseName
   else if Assigned(fDataLink.DataSet)
          then X.DatabaseName:=TDBDataSet(fDataLink.DataSet).DatabaseName;

   X.SQL.Text:=S;
   X.Open;
   if (Mask <> '') and X.EOF then
   begin
      Mask:='';
      goto L0;
   end;

   if (X.RecordCount > 1) then
   begin
      while not X.EOF
      and (X.Fields[0].AsString<>Value)
      do X.Next;
      if X.Fields[0].AsString <> Value then X.First;
      X.Fields[0].Visible:=false;
      if YSelForm.Run(Self,X) = mrOK then Selected:=true;
   end else Selected:=true;

   if Selected
      then Value:=X.Fields[0].AsString
      else Reload;

   X.Close;
   if Selected and Assigned(fAfterSelect)
      then fAfterSelect(Self);
end;

procedure   TYDBLookUp.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var R : TRect;
begin
   R:=BtnRect(0);
   if (X>R.Left) and (X<R.Right)
   and (Y>R.Top) and (Y<R.Bottom)
       then DoClickSel;

   R:=BtnRect(1);
   if (X>R.Left) and (X<R.Right)
   and (Y>R.Top) and (Y<R.Bottom)
       then DoClickAdd;
end;

procedure   TYDBLookUp.DoClickSel;
begin
   DrawBtn(0,true,'>');
   sleep(150);
   DrawBtn(0,false,'>');
   if Assigned(fOnClickSel)
      then fOnClickSel(Self)
      else SelectRecord;
end;

procedure   TYDBLookUp.DoClickAdd;
begin
  DrawBtn(1,true,'+');
  sleep(150);
  DrawBtn(1,false,'+');
  if Assigned(fOnClickAdd)
     then fOnClickAdd(Self);
end;

//****************************************************
procedure TYDBLookUp.xKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = '#' then Key:=#0;
  if (Key in [' '..#255]) and (Edit.Text=fText)
     then Edit.Text:='';
end;

//****************************************************
procedure TYDBLookUp.xKeyDown(Sender : TObject; var Key: Word; Shift: TShiftState);
begin
   if (Key = vk_Down) or (Key= vk_F4) then
   begin
     Key:=0;
     DoClickSel;
   end else
   if (Key = vk_Insert) then
   begin
     Key:=0;
     DoClickAdd;
   end else
   if (Key = vk_Escape) or (Key = vk_Up) then
   begin
     Key:=0;
     fDataLink.Reset;
     Reload;
   end;
end;

function  TYDBLookUp.GetDataField : string;
begin
  result:=fDataLink.FieldName;
end;

procedure TYDBLookUp.SetDataField(aDataField : string);
begin
  fDataLink.FieldName:=aDataField;
end;

procedure TYDBLookUp.SetDatabaseName(const V : string);
begin
  FDatabaseName := V;
end;

procedure TYDBLookUp.SetSQLShow(V : TStrings);
begin
   fSQLshow.Assign(V);
end;

procedure TYDBLookUp.SetSQLFind(V : TStrings);
begin
   fSQLFind.Assign(V);
end;

function  TYDBLookUp.ShowDataOf(V : string) : string;
var N : integer;
    S : string;
    X : TQuery;
begin
  result:='';

  if pos('#',V) > 0
     then raise exception.create('Invalid character "#" in input parameter');

  V:=''''+V+'''';

  S:=fSQLShow.Text;
  N:=Pos('#',S);
  while N > 0 do
  begin
    delete(S,N,1);
    insert(V,S,N);
    N:=Pos('#',S);
  end;

  X := TQuery.Create(nil);
  if fDatabaseName <> ''
  then X.DatabaseName:=fDatabaseName
  else if Assigned(fDataLink.DataSet)
          then X.DatabaseName:=TDBDataSet(fDataLink.DataSet).DatabaseName;
  X.SQL.Add(S);
  try
    X.Open;
    result:=UnpackFields(X,false);
  finally
    X.Free;
  end;
  result:=Trim(result);
end;

procedure   TYDBLookUp.Reload;
begin
  if (Trim(fSQLShow.Text) <> '')
     then Edit.Text:=ShowDataOf(fValue)
     else Edit.Text:='';
  fText:=Edit.Text;
end;

procedure TYDBLookUp.SetValue(V : string);
begin
  if (fLockChange=0)
  and Assigned(fDataLink.Field) then
  begin
    inc(fLockChange);
    try
      if fDataLink.Field.AsString <> V then
      begin
         fDataLink.DataSet.Edit;
         fDataLink.Field.AsString:=V;
      end;
      fValue:=fDataLink.Field.AsString;
    finally
      dec(fLockChange);
    end;
  end else fValue:=V;

  if (fLastValue <> fValue)
  then Reload;

  if (fLastValue <> fValue)
  and (fLockChange=0)
  and assigned(fOnChange)
  then fOnChange(Self);

  fLastValue:=fValue;
end;

procedure TYDBLookUp.DataChange(Sender : TObject);
begin
  if Assigned(fDataLink.Field)
  and not (csDesigning in ComponentState)
  then Value:=fDataLink.Field.AsString;
end;

initialization

  YSelForm  :=  TYSelForm.Create(nil);

finalization

  YSelForm.Release;

end.

{-----------------------------------------------------------------
This component represents a text that identifies a set of
field values in one dataset with a corresponding set of values
from another dataset. It work like TDBLookUp, but has additional
features and very different mechanism to selecting corresponding data.

It has two dynamically created SQL: SQLshow and SQLfind.

SQLshow is intended to show corresponding data.
The samples of SQLshow:

----- one table ------
    SELECT FIELD1,FIELD2
    FROM TABLENAME
    WHERE FIELD3 = #

----- two tables join ------
    SELECT FIELD1,FIELD2,FIELD3
    FROM TABLE1,TABLE2
    WHERE TABLE2.ID = #
    AND TABLE2.ID_TABLE1 = TABLE1.ID
-----
"#" character will be automatically replaced by value of DataField.
Single quotas will be added before and after #.
FIELD1,FIELD2,(FIELD3) will apear in control and changed every time
when DataField just changed.

SQLfind is intended to select DataField value using SQL query.
The sample of SQLfind:
------ one table -----
    SELECT VALUEFIELD,FIELD1,FIELD2
    FROM TABLE1
    WHERE UPPER(FIELD1) LIKE "#"

"#" character will be automatically replaced by value typed by
user in this control. And may be used in search condition.

The first field in fields roster must be the field which value
will be assigned to DataField. Rest of fields whill be shown in
dropdown YDBGrid. To show first field in dropdown grid it is
necessary to add it to fields roster twice:
SELECT VALUEFIELD,VALUEFIELD,FIELD1,FIELD2


PROPERTIES to adjust dropdown YDBGrid is:

SQLFindMode property [fmMatch,fmStart,fmInclude]
    fmMatch - exactly match (WHERE FIELD1 = "ABC")
    fmStart - search by first characters, automatically add "%"
              to end of typed text. (WHERE FIELD1 LIKE "ABC%")
    fmInclude - search by substring, automatically add "%"
              before and after type text. (WHERE FIELD1 LIKE "%ABC%")
    for numeric fields usually used fmMatch.

SQLFindUpCase property
    it will be convert text typed by user to upper case just
    before "#" will be replaced by typed text.

AdjustCols property
    maximum widths of dropped down YDBGrid columns will be
    automatically adjusted to max lenghts of values in this grids.
    It may to decrease the speed of the droppdown grid appearence.


EVENTS

OnChange   - occurs when value in DataField changed
OnClickAdd - occurs when user clicked [+] button. [+] button
             will not been shown if this event not assigned.
OnClickSel - occurs when user click dropdown button. YDBGrid will
             NOT automatically dropped down if this method is
             assigned, in this case user must call DoClickSel
             itself. It must be keeped alone if user do not planed
             to write its own code to support dropdown method.
AfterSelect- occurs just after DataField value was selected from
             dropdown list.


TIPS and TRICKS

It doesnt show any text in designtime.

If user type some text in control and then control loose focus,
([TAB] key pressed) then dropdown grid will shown automatically.

DataField may not be assigned. It may work standalone. Show only.
Use public property VALUE to get and put value to/from control.
VALUE is string.

If DataField assigned, DatabaseName may not be assigned, SQLshow
and SQLfind will use the same database as DataField.DataSorce.DataSet.

User must decide to use double quotas in SQLfind around "#" char or not.
It not inserted automatically.

    Example SQLfind for search by numeric value:
        WHERE NUMFIELD = #

    Example SQLfind for search by text value:
        WHERE STRFIELD LIKE "#"
        WHERE UPPER(STRFIELD) LIKE "#"

Key control
    [F4], [Down] - drop down YDBGrid.
    [INS]        - OnClickAdd occurs.
    [ESC]        - cancel drop down.

Typing in dropped down grid will automatically locate
record by first typed characters in field.

Enabled = false - hide all buttons.


*************************
06-apr-2000 testing start
------------------------------------------------------------------}
